<?php

namespace App\Http\Controllers\Api\Dentist;

use App\Models\Cart;
use App\Models\Product;
use Illuminate\Http\Request;
use App\Traits\HasCrudActions;
use App\Http\Controllers\Controller;
use App\Http\Resources\Dentist\CartItemsResource;
use App\Http\Resources\Api\Dentist\ProductResource;

class CartController extends Controller
{
    use HasCrudActions;
    public function addToCart(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
        ]);
    
        $userId = auth()->id();
        $cart = Cart::firstOrCreate(['user_id' => $userId]);
    
        $product = Product::findOrFail($request->product_id);
        $cartItem = $cart->items()->where('product_id', $request->product_id)->first();
    
        if ($cartItem) {
            $cartItem->update([
                'quantity' => $cartItem->quantity + $request->quantity,
                'price' => $product->price,
            ]);
        } else {
            $cart->items()->create([
                'product_id' => $request->product_id,
                'quantity' => $request->quantity,
                'price' => $product->price,
            ]);
        }
    
        // Update total price and total quantity in the cart
        $cart->update([
            'total_price' => $cart->items->sum(fn($item) => $item->quantity * $item->price),
            'total_quantity' => $cart->items->sum('quantity'),
        ]);
    
        return $this->apiResponse('200', __('success'), 'Added to cart', $cart->load('items.product'));
    }
    



    // Update item quantity in the cart
    public function updateCart(Request $request)
{
    $request->validate([
        'product_id' => 'required|exists:products,id',
        'quantity' => 'required|integer|min:1',
    ]);

    $cart = Cart::where('user_id', auth()->id())->first();

    if (!$cart) {
        return $this->apiResponse('400', __('error'), 'Cart not found');
    }

    $cartItem = $cart->items()->where('product_id', $request->product_id)->first();

    if ($cartItem) {
        $cartItem->update(['quantity' => $request->quantity]);

        // Update total price and total quantity in the cart
        $cart->update([
            'total_price' => $cart->items->sum(fn($item) => $item->quantity * $item->price),
            'total_quantity' => $cart->items->sum('quantity'),
        ]);

        return $this->apiResponse('200', __('success'), 'Cart updated', $cart->load('items.product'));
    }

    return $this->apiResponse('400', __('error'), 'Product not found in cart');
}




    // Remove item from cart
    public function removeFromCart($productId)
    {
        $cart = Cart::where('user_id', auth()->id())->first();
    
        if (!$cart) {
            return $this->apiResponse('400', __('error'), 'Cart not found');
        }
    
        $cart->items()->where('product_id', $productId)->delete();
    
        // Update total price and total quantity in the cart
        $cart->update([
            'total_price' => $cart->items->sum(fn($item) => $item->quantity * $item->price),
            'total_quantity' => $cart->items->sum('quantity'),
        ]);
    
        return $this->apiResponse('200', __('success'), 'Product removed from cart', $cart->load('items.product'));
    }
    

    // Retrieve cart
    public function viewCart()
    {
        $cart = Cart::with('items.product')->where('user_id', auth()->id())->first();
    
        if (!$cart) {
            return $this->apiResponse('200', __('success'), 'Cart is empty', [
                'products' => [],
                'cart_total' => 0,
                'items_count' => 0,
            ]);
        }
    
        return $this->apiResponse('200', __('success'), '', [
            'products' => CartItemsResource::collection($cart->items),
            'cart_total' => $cart->total_price,
            'items_count' => $cart->total_quantity,
        ]);
    }
    


}
