<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use App\Models\RFQ;
use App\Models\User;
use App\Models\RfqBid;
use App\Models\Setting;
use App\Services\FcmService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class RfqCheck extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:rfq-check';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Execute the console command.
     */
    public function handle(FcmService $fcmService)
    {
        Log::channel('rfq')->info('RfqCheck command started'); // Log start of the command

        try {
            // Get RFQ deadline from settings
            $rfq_deadline = Setting::select('rfq_time')->first();
            $current_time = now()->format('H:i');

            Log::channel('rfq')->info('RFQ deadline from settings', ['rfq_time' => $rfq_deadline->rfq_time, 'current_time' => $current_time]);

            // Check if current time matches the deadline
            if (Carbon::parse($rfq_deadline->rfq_time)->format('H:i') == $current_time) {
                $bids = RfqBid::where('is_approved', 1)->get();
                Log::channel('rfq')->info('Approved bids count', ['count' => $bids->count()]);

                foreach ($bids as $bid) {
                    // Find the RFQ associated with the bid
                    $rfq = RFQ::where(['id' => $bid->rfq_id, 'vendor_id' => null])->first();

                    // If RFQ is found, update the vendor
                    if ($rfq) {
                        $rfq->vendor_id = $bid->vendor_id;
                        $rfq->status = 'won';
                        $rfq->save();
                        Log::channel('rfq')->info('RFQ updated', ['rfq_id' => $rfq->id, 'vendor_id' => $rfq->vendor_id]);

                        // Notify the vendor about their winning bid
                        $vendor = User::find($rfq->vendor_id); // Assuming vendors are stored in the `users` table
                        if ($vendor && $vendor->fcmTokens()->exists()) {
                            $fcmService->send(
                                $vendor->fcmTokens[0]->device_token,
                                __('Bid Won'),
                                __('Congratulations! Your bid for the RFQ #:rfq_id has been accepted.', ['rfq_id' => $rfq->id]),
                                [
                                    'rfq_id' => (string) $rfq->id,
                                    'action' => 'bid_won'
                                ],
                                $vendor
                            );

                            Log::channel('rfq')->info('Notification sent to vendor', ['vendor_id' => $vendor->id, 'rfq_id' => $rfq->id]);
                        } else {
                            Log::channel('rfq')->warning('No FCM token found for vendor', ['vendor_id' => $rfq->vendor_id]);
                        }
                    } else {
                        Log::channel('rfq')->warning('RFQ not found for bid', ['rfq_id' => $bid->rfq_id]);
                    }
                }

                Log::channel('rfq')->info('Bids successfully processed');
                return "تم التصديق بالمزايدات";
            }

            Log::channel('rfq')->info('No matching RFQ time found, command completed');
            return "لسه";

        } catch (\Exception $e) {
            // Log the exception details
            Log::channel('rfq')->error('Error in RfqCheck command', ['message' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);

            // Return or throw the exception if needed
            return "Error occurred: " . $e->getMessage();
        }
    }

}
