<?php

namespace App\Http\Controllers\Api\Admin;

use App\Models\Admin;
use App\Models\Product;
use App\helpers\Attachment;
use Illuminate\Http\Request;
use App\Traits\HasCrudActions;
use App\Imports\ProductsImport;
use App\Events\UserNotification;
use App\Traits\Notificationable;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use App\Notifications\PushNotification;
use Illuminate\Support\Facades\Notification;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Requests\Api\Admin\ProductRequest;
use App\Http\Resources\Api\Admin\ProductResource;

class ProductController extends Controller
{
    use HasCrudActions, Notificationable;
    protected $model = Product::class;

    protected $resource = ProductResource::class;

    protected $request = ProductRequest::class;

    public function index()
    {
        $modelResource = $this->search();

        return $this->paginatedResponse($modelResource->with(['mainCategory', 'subCategory', 'attachmentRelation'])->orderBy('id', 'DESC'), ProductResource::class);
    }

    public function show(string $id)
    {
        $record = $this->model::find($id)->load(['mainCategory', 'subCategory', 'attachmentRelation', 'specs']);

        if (!$record) {
            return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
        }

        $recordResource = new $this->resource($record);

        return $this->apiResponse(Response::HTTP_OK, __('success'), '', $recordResource);
    }

    public function store()
    {

        try {
            $request = app($this->request);

            DB::beginTransaction();

            // Save product data
            $recordData = $request->except('image', 'specs');
            $record = $this->model::create($recordData);
            $record->save(); // Save the updated record to persist the user_id
            if ($request->hasFile('image')) {
                Attachment::addAttachment($request->file('image'), $record, 'products/images', ['save' => 'original', 'usage' => 'image']);
            }

            if ($request->has('specs')) {
                foreach ($request->specs as $spec) {
                    $record->specs()->create([
                        'key' => $spec['key'],
                        'value' => $spec['value'],
                    ]);
                }
            }

            DB::commit();

            // Return response
            $recordResource = new $this->resource($record);
            return $this->apiResponse(Response::HTTP_CREATED, __('success'), __('created successfully'), $recordResource);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->log(0, $this->model, $e->getMessage(), $e);
            return $this->errorMessage($e);
        }
    }

    public function update(string $id)
    {
        try {
            $request = app($this->request);

            $record = $this->model::find($id);

            if (!$record) {
                return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
            }

            DB::beginTransaction();
            $record->update($request->except('image'));
             // Attach media if provided
             if ($request->hasFile('image')) {
                Attachment::updateAttachment($request->file('image'), $record->image,$record , 'products/images', ['save' => 'original', 'usage' => 'image']);
            }

        if ($request->has('specs')) {
            foreach ($request->specs as $spec) {
                $record->specs()->updateOrCreate(
                    ['key' => $spec['key']], // The unique identifier to find the existing spec
                    ['value' => $spec['value']] // Fields to update or create
                );
            }
        }

            DB::commit();

            $recordResource = new $this->resource($record);

            return $this->apiResponse(Response::HTTP_OK, __('success'), __('Updated Successfully!'), $recordResource);
        } catch (\Exception $e) {

            DB::rollBack();

            $this->log(0, $this->model, $e->getMessage(),$e);

            return $this->errorMessage($e);
        }
    }

    public function SendAccessRequest()
    {
        $user = Admin::find(1);
        Notification::send($user, new PushNotification([
            'title' => 'dasdas',
            'message' => 'dasd',
        ]));
        return 'success';
    }

    public function getVendorProducts($vendorId)
    {
        $products = Product::where('user_id', $vendorId)->get();

        return $this->apiResponse(Response::HTTP_OK, __('success'), __('Updated Successfully!'), ProductResource::collection($products));

        
    }


    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls',
        ]);

        Excel::import(new ProductsImport, $request->file('file'));
        return $this->apiResponse(Response::HTTP_OK, __('success'), __('Products imported successfully.'));

    }
}
