<?php

namespace App\Exports;

use App\Models\RFQ;
use App\Models\Order;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;

class OrdersExport implements FromCollection, WithHeadings, WithMapping, WithEvents
{
        protected $type;

    /**
     * Constructor to accept the order type.
     *
     * @param string $type
     */
    public function __construct(string $type)
    {
        $this->type = $type;
    }

    /**
     * Fetch the collection based on the type.
     *
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        if ($this->type === 'direct_order') {
            return RFQ::with(['product', 'branch', 'user', 'vendor'])
                ->where('order_type', 'direct_order')
                ->get();
        }

        if ($this->type === 'rfq_order') {
            return RFQ::with(['product', 'branch', 'user', 'vendor', 'rfqItems', 'specs'])
            ->where('order_type', 'rfq_order')

                ->get();
        }

        // Default case for 'orders': Get both RFQs and Orders
        return RFQ::with(['product', 'branch', 'user', 'vendor'])
            ->get()
            ->merge(
                RFQ::with(['product', 'branch', 'user', 'vendor', 'rfqItems', 'specs'])->get()
            );
    }

    /**
     * Define the headings for the export.
     *
     * @return array
     */
    public function headings(): array
    {
        $headings = [
            'Quantity',
            'Status',
            'Price',
            'Type',
            'Created At',
            'Branch Location',
            'User Owner Name',
            'Vendor Owner Name',
        ];
    
        if ($this->type === 'rfq_order') {
            array_splice($headings, 5, 0, 'Product Name'); // Insert "Product Name" after "Created At"
        }
    
        return $headings;
    }
    
    /**
     * Map the data to be exported.
     *
     * @param mixed $order
     * @return array
     */
    public function map($order): array
    {
        return [
            $order->qty ?? null,
            $order->status ?? null,
            $order->price ?? null,
            $order->order_type ?? 'rfq_order',
            $order->created_at->format('Y-m-d H:i:s'),
            $order->branch->location ?? null,
            $order->user->owner_name ?? null,
            $order->vendor->owner_name ?? null,
            $this->type === 'rfq_order' ? $order->product->name ?? null : null, // Include product name only for rfq_order

        ];
    }

    /**
     * Register events to auto-size columns.
     *
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet;

                // Auto-size each column
                foreach (range('A', 'I') as $column) {
                    $sheet->getDelegate()->getColumnDimension($column)->setAutoSize(true);
                }
            },
        ];
    }
}
