<?php

namespace App\Http\Controllers\Api\vendor;

use App\Models\Admin;
use App\Models\Product;
use App\helpers\Attachment;
use App\Services\FcmService;
use Illuminate\Http\Request;
use App\Traits\HasCrudActions;
use App\Events\UserNotification;
use App\Traits\Notificationable;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Notifications\PushNotification;
use Illuminate\Support\Facades\Notification;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Requests\Api\vendor\ProductRequest;
use App\Http\Resources\Api\vendor\ProductResource;

class ProductController extends Controller
{
    use HasCrudActions, Notificationable;
    protected $model = Product::class;

    protected $resource = ProductResource::class;

    protected $request = ProductRequest::class;

    public function index()
    {
        $modelResource = $this->search();

        return $this->paginatedResponse($modelResource->with(['mainCategory', 'subCategory', 'attachmentRelation'])->orderBy('id', 'DESC'), ProductResource::class);
    }

    public function show(string $id)
    {
        $record = $this->model::find($id)->load(['mainCategory', 'subCategory', 'attachmentRelation', 'specs']);

        if (!$record) {
            return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
        }

        $recordResource = new $this->resource($record);

        return $this->apiResponse(Response::HTTP_OK, __('success'), '', $recordResource);
    }

    public function store()
    {

        try {
            $request = app($this->request);

            DB::beginTransaction();

            // Save product data
            $recordData = $request->except('image', 'specs');
            $record = $this->model::create($recordData);
            $record->user_id = Auth::id();
            $record->save(); // Save the updated record to persist the user_id

            if ($request->hasFile('image')) {
                Attachment::addAttachment($request->file('image'), $record, 'users/profile', ['save' => 'original', 'usage' => 'image']);
            }

            if ($request->has('specs')) {
                foreach ($request->specs as $spec) {
                    $record->specs()->create([
                        'key' => $spec['key'],
                        'value' => $spec['value'],
                    ]);
                }
            }

            DB::commit();

            // Return response
            $recordResource = new $this->resource($record);
            return $this->apiResponse(Response::HTTP_CREATED, __('success'), __('created successfully'), $recordResource);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->log(0, $this->model, $e->getMessage(), $e);
            return $this->errorMessage($e);
        }
    }

    public function SendAccessRequest(FcmService $fcmService)
    {
        $user = Auth::user();
    
        // Update user's product access status if not already 'pending'
        if ($user->product_access != 'pending') {
            $user->update([
                'product_access' => 'pending'
            ]);
        }
    
        // Prepare data for the notification
        $data = [
            'user_id' => (string) $user->id,
            'action' => 'access_request'
        ];
    
        // Fetch all admins with their FCM tokens
        $admins = Admin::with('fcmTokens')->get();
    
        $successCount = 0;
        $failureCount = 0;
    
        // Loop through all admins
        foreach ($admins as $admin) {
            foreach ($admin->fcmTokens as $token) {
                if ($token->device_token) {
                    $fcmResponse = $fcmService->send(
                        $token->device_token,
                        'Access Request Pending',
                        "{$user->owner_name} has requested access to products.",
                        $data,
                        $user
                    );
    
                    // Track success and failure counts
                    if ($fcmResponse->successful()) {
                        $successCount++;
                    } else {
                        $failureCount++;
                    }
                }
            }
        }
    
        // Return appropriate response
        if ($successCount > 0) {
            return $this->apiResponse(
                Response::HTTP_CREATED,
                __('success'),
                __('Notification sent successfully')
            );
        } else {
            return $this->apiResponse(
                Response::HTTP_INTERNAL_SERVER_ERROR,
                __('error'),
                __('Failed to send notifications to admins')
            );
        }
    }
    
    
}
